<?php
// verify.php - Server-side Turnstile verification

// CORS headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');
header('Content-Type: application/json');
header('X-Robots-Tag: noindex, nofollow, nosnippet, noarchive');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    exit(0);
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Turnstile secret key
$TURNSTILE_SECRET_KEY = "0x4AAAAAAAhkfDvi_knYUplb1Ga8g_OYNO0";

// Rate limiting
session_start();
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$rate_limit_key = 'verify_attempts_' . md5($ip);

if (!isset($_SESSION[$rate_limit_key])) {
    $_SESSION[$rate_limit_key] = ['count' => 0, 'time' => time()];
}

$attempts = $_SESSION[$rate_limit_key];
if (time() - $attempts['time'] > 3600) {
    // Reset after 1 hour
    $_SESSION[$rate_limit_key] = ['count' => 0, 'time' => time()];
    $attempts = $_SESSION[$rate_limit_key];
}

if ($attempts['count'] >= 10) {
    http_response_code(429);
    echo json_encode(['success' => false, 'error' => 'Rate limit exceeded']);
    exit;
}

// Increment attempt counter
$_SESSION[$rate_limit_key]['count']++;

// Get JSON input
$json = file_get_contents('php://input');
$data = json_decode($json, true);

if (!$data || !isset($data['token']) || !isset($data['email'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid request data']);
    exit;
}

$token = $data['token'];
$email = filter_var($data['email'], FILTER_VALIDATE_EMAIL);

if (!$email) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid email format']);
    exit;
}

// Bot detection
$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
$bot_patterns = '/bot|crawl|spider|scraper|python|curl|wget|facebookexternalhit|whatsapp|telegram/i';

if (preg_match($bot_patterns, $user_agent)) {
    http_response_code(403);
    echo json_encode(['success' => false, 'error' => 'Access denied']);
    exit;
}

// Verify Turnstile token with Cloudflare
$verify_url = 'https://challenges.cloudflare.com/turnstile/v0/siteverify';
$verify_data = [
    'secret' => $TURNSTILE_SECRET_KEY,
    'response' => $token,
    'remoteip' => $ip
];

$options = [
    'http' => [
        'header' => "Content-type: application/x-www-form-urlencoded\r\n",
        'method' => 'POST',
        'content' => http_build_query($verify_data),
        'timeout' => 10
    ]
];

$context = stream_context_create($options);
$result = @file_get_contents($verify_url, false, $context);

if ($result === false) {
    error_log("Turnstile verification failed - network error for IP: $ip");
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Verification service unavailable']);
    exit;
}

$response = json_decode($result, true);

if (!$response || !isset($response['success'])) {
    error_log("Turnstile verification failed - invalid response for IP: $ip");
    http_response_code(500);
    echo json_encode(['success' => false, 'error' => 'Invalid verification response']);
    exit;
}

if ($response['success']) {
    // Reset rate limit on successful verification
    $_SESSION[$rate_limit_key]['count'] = 0;
    
    // Log successful access (hash sensitive data for privacy)
    $log_data = [
        'timestamp' => date('Y-m-d H:i:s'),
        'email_hash' => hash('sha256', $email),
        'ip_hash' => hash('sha256', $ip),
        'user_agent' => substr($user_agent, 0, 100)
    ];
    
    // Log to file (optional)
    error_log("Document access granted: " . json_encode($log_data));
    
    // Return success
    echo json_encode([
        'success' => true,
        'message' => 'Verification successful',
        'timestamp' => time()
    ]);
} else {
    // Log failed verification
    $error_codes = $response['error-codes'] ?? [];
    error_log("Turnstile verification failed for IP $ip. Errors: " . implode(', ', $error_codes));
    
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => 'Verification failed',
        'details' => $error_codes
    ]);
}
?>